//*************************************************************************************************
//
//	Description:
//		test_foliage.fx - Quick foliage test shader for Sven.
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     15/09/2006  0.1						Created
//		TNettleship			10/19/2006	0.2						Removed texture mapping info params from display.
//		TNettleship			11/21/2006	0.3						Added support for deferred rendering.
//		TMann						11/24/2006	1.4						Added _WIN_GL_ support
//		TMann						11/27/2006	1.5						Texture filtering for GL
//		TMann						12/11/2006	1.6						PS3 version
//		TNettleship			05/03/2007	2.0						Specialisation + added shadow override technique for basic_translucent.
//		TNettleship			07/10/2007	2.01					Added support for vertex alpha.
//		TNettleship     07/24/2007  2.02          Made sure samplers aren't using anisotropic filtering.
//		TNettleship     08/17/2007  2.03					Removed texture bias.
//	<TABLE>
//
//*************************************************************************************************

#include "stddefs.fxh"


//-----------------------------------------------------------------------
//
// Input parameters
//


//
// Transforms
//

#if defined( _3DSMAX_ ) || defined(USE_WVP_CONSTANT)
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;


//
// Channel mappings (max only)
//

#ifdef _3DSMAX_
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// Vertex alpha channel (max presents it seperately for no good reason)
int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = -2;
	bool ColorChannel = true;
	bool export = false;
> = 0;

#endif

//
// Textures
//

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Texture";
	bool appEdit = true;
>;




//
// Lighting
//

float4 selfIlluminationColour : Ambient				// Self illumination colour
<
	string UIName = "Selfillum Colour";
	string UIType = "ColorSwatch";
	bool appEdit = true;
> = {0.25f, 0.25f, 0.25f, 0.0f};



//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};


//-----------------------------------------------------------------------
//
// Functions
//



//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position : POSITION;														// Object space position
#ifdef _3DSMAX_
	float3 colour   : TEXCOORD1;													// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
	float3 alpha		: TEXCOORD2;													// Vertex alpha
#else
	float4 colour   : COLOR0;															// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord
#endif
};


// Output structures
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD1;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
};


struct SHADOWGEN_VSOUTPUT
{
	float4 position			: POSITION;													// View-coords position
	float2 texCoord			: TEXCOORD0;												// UV coords for texture channel 0
	float4 shadowCoord	: TEXCOORD1;												// worldpos for shadow transformation
};

struct ZPRIMEDOF_VSOUTPUT
{
	float4 position			: POSITION;													// View-coords position
	float2 texCoord			: TEXCOORD0;												// UV coords for texture channel 0
	float4 coords				: TEXCOORD1;
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT TestFoliageVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
	_output.texCoord = _input.texCoord;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	return _output;
}



SHADOWGEN_VSOUTPUT TestFoliageShadowGenVertexShader( VSINPUT _input )
{
	SHADOWGEN_VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
	_output.texCoord = _input.texCoord;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Clamp geometry that is behind the camera to the near plane so that it still renders.
	// We use an orthogonal projection, so doing this will not distort the shadow caster.
	CLAMP_SHADOW_Z;

	OUTPUT_SHADOW_COORDS;

	return _output;
}

ZPRIMEDOF_VSOUTPUT TestFoliageZPrimeDOFVertexShader( VSINPUT _input )
{
	ZPRIMEDOF_VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
	_output.texCoord = _input.texCoord;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );
	_output.coords = _output.position;

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

// Input structures
struct PSINPUT
{
	float4 colour		: TEXCOORD1;						// Vertex colour
	float2 texCoord : TEXCOORD0;					// UV coords for texture channel 0
};

struct SHADOWGEN_PSINPUT
{
	float2 texCoord : TEXCOORD0;
	float4 shadowCoord : TEXCOORD1;
};


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

PSOUTPUT TestFoliageFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	// Read textures
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	// Factor vertex alpha into the diffuse alpha
	diffuseTexColour.a *= _input.colour.a;

	// Calculate pixel colour
	float4 colour;
	colour  = float4( selfIlluminationColour.xyz, 1.0f ) * diffuseTexColour;

	// Rescale alpha [0..0.25) --> 0.0, [0.25..1.0] --> [0.0..1.0]
	colour.w = saturate( diffuseTexColour.w - 0.25f ) * ( 1.0f / 0.75f );
	
	_output.Colour=CalculateOutputPixel(colour);

	return _output;
}


PSOUTPUT TestFoliageShadowGenFragmentShader( SHADOWGEN_PSINPUT _input )
{
	PSOUTPUT _output;

	// Read texture
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	_output.Colour.xyz = _input.shadowCoord.z / _input.shadowCoord.w;

	// Rescale alpha [0..0.25) --> 0.0, [0.25..1.0] --> [0.0..1.0]
	_output.Colour.w = saturate( diffuseTexColour.w - 0.25f ) * ( 1.0f / 0.75f );

	return _output;
}

PSOUTPUT TestFoliageZPrimeDOFFragmentShader( ZPRIMEDOF_VSOUTPUT _input )
{
	PSOUTPUT _output;

	// Read texture
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	_output.Colour.xyz = _input.coords.z / _input.coords.w;

	// Rescale alpha [0..0.25) --> 0.0, [0.25..1.0] --> [0.0..1.0]
	_output.Colour.w = saturate( diffuseTexColour.w - 0.25f ) * ( 1.0f / 0.75f );

	return _output;
}



//-----------------------------------------------------------------------
//
// Technique(s)
//

technique test_foliage
<
	bool supportsSpecialisedLighting = false;
	bool preservesGlobalState = true;
	string normalBehaviour			= "ERMB_RENDER";
	string normalTechnique			= "test_foliage";
	int    normalDeferredID			= 2;
	string zprimeBehaviour			= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour		= "ERMB_RENDER";
	string zprimeDOFTechnique		= "_test_foliage_zprime_dof";
	int    zprimeDOFDeferredID	= 0;
	string shadowGenBehaviour		= "ERMB_RENDER";
	string shadowGenTechnique		= "_test_foliage_shadowgen";
	int    shadowGenDeferredID	= 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = true;
		string DestBlend = "INVSRCALPHA";
		string SrcBlend = "SRCALPHA";
		string BlendOp = "ADD";
		bool AlphaTestEnable = true;
    int AlphaRef = 0x00000040;
    string AlphaFunc = "GreaterEqual";
    string CullMode = "None";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = 1;
		ZWriteEnable = true;
		AlphaBlendEnable = true;
		DestBlend = INVSRCALPHA;
		SrcBlend = SRCALPHA;
		BlendOp = ADD;
		AlphaTestEnable = true;
    AlphaRef = 0x00000040;
    AlphaFunc = GreaterEqual;
    CullMode = None;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx TestFoliageVertexShader();
		PixelShader = compile sce_fp_rsx TestFoliageFragmentShader();
#else		
		VertexShader = compile vs_3_0 TestFoliageVertexShader();
		PixelShader = compile ps_3_0 TestFoliageFragmentShader();
#endif		
	}
}


technique _test_foliage_shadowgen
{
	pass Pass0
	{
		SHADOWMAP_STATES_ALPHATEST( 128 )
#if defined (_PS3_)
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
		VertexShader = compile sce_vp_rsx TestFoliageShadowGenVertexShader();
		PixelShader = compile sce_fp_rsx TestFoliageShadowGenFragmentShader();
#elif defined (_XBOX)
		VertexShader = compile vs_3_0 TestFoliageShadowGenVertexShader();
		PixelShader = compile ps_3_0 TestFoliageShadowGenFragmentShader();
#else		
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
		VertexShader = compile vs_3_0 TestFoliageShadowGenVertexShader();
		PixelShader = compile ps_3_0 TestFoliageShadowGenFragmentShader();
#endif
	}
}

technique _test_foliage_zprime_dof
{
	pass Pass0
	{
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
		AlphaTestEnable = true;
#if defined (_PS3_)
		AlphaFunc = int2(GEqual, 0x40);
		CullFaceEnable = false;
		VertexShader = compile sce_vp_rsx TestFoliageZPrimeDOFVertexShader();
		PixelShader = compile sce_fp_rsx TestFoliageZPrimeDOFFragmentShader();
#else		
		AlphaRef = 0x40;
		AlphaFunc = GreaterEqual;
		CullMode = None;
		VertexShader = compile vs_3_0 TestFoliageZPrimeDOFVertexShader();
		PixelShader = compile ps_3_0 TestFoliageZPrimeDOFFragmentShader();
#endif
	}
}
